--[[------------------------------------------------
	-- Love Frames - A GUI library for LOVE --
	-- Copyright (c) 2013 Kenny Shields --
--]]------------------------------------------------

local imgmanip = require("Engine/ImageManipulator")

-- skin table
local skin = {}

-- skin info (you always need this in a skin)
skin.name = "Menu"
skin.author = "Berbece Man"
skin.version = "1.0"

local smallfont = love.graphics.newFont("Lib/UI/Misc/AlteHaas.ttf",15)
local imagebuttonfont = love.graphics.newFont("Lib/UI/Misc/AlteHaas.ttf",15)
local displayTextFont = love.graphics.newFont("Lib/UI/Misc/AlteHaas.ttf",18)
local bordercolor = {143, 143, 143, 255}

-- controls
skin.controls = {}

-- frame
skin.controls.frame_body_color                      = {232, 232, 232, 255}
skin.controls.frame_name_color                      = {255, 255, 255, 255}
skin.controls.frame_name_font                       = smallfont
skin.controls.displayTextFont = displayTextFont

-- button
skin.controls.button_text_down_color                = {255, 255, 255, 255}
skin.controls.button_text_nohover_color             = {0, 0, 0, 200}
skin.controls.button_text_hover_color               = {255, 255, 255, 255}
skin.controls.button_text_nonclickable_color        = {0, 0, 0, 100}
skin.controls.button_text_font                      = smallfont

-- imagebutton
skin.controls.imagebutton_text_down_color           = {255, 255, 255, 255}
skin.controls.imagebutton_text_nohover_color        = {255, 255, 255, 200}
skin.controls.imagebutton_text_hover_color          = {255, 255, 255, 255}
skin.controls.imagebutton_text_font                 = imagebuttonfont

-- closebutton
skin.controls.closebutton_body_down_color           = {255, 255, 255, 255}
skin.controls.closebutton_body_nohover_color        = {255, 255, 255, 255}
skin.controls.closebutton_body_hover_color          = {255, 255, 255, 255}

-- progressbar
skin.controls.progressbar_body_color                = {255, 255, 255, 255}
skin.controls.progressbar_text_color                = {0, 0, 0, 255}
skin.controls.progressbar_text_font                 = smallfont

-- list
skin.controls.list_body_color                       = {232, 232, 232, 0}

-- scrollarea
skin.controls.scrollarea_body_color                 = {200, 200, 200, 0}

-- scrollbody
skin.controls.scrollbody_body_color                 = {0, 0, 0, 0}

-- panel
skin.controls.panel_body_color                      = {0, 0, 0, 0}

-- tabpanel
skin.controls.tabpanel_body_color                   = {0, 0, 0, 0}

-- tabbutton
skin.controls.tab_text_nohover_color                = {0, 0, 0, 200}
skin.controls.tab_text_hover_color                  = {255, 255, 255, 255}
skin.controls.tab_text_font                         = smallfont

-- multichoice
skin.controls.multichoice_body_color                = {240, 240, 240, 255}
skin.controls.multichoice_text_color                = {100, 100, 100, 255}
skin.controls.multichoice_text_font                 = smallfont

-- multichoicelist
skin.controls.multichoicelist_body_color            = {240, 240, 240, 200}

-- multichoicerow
skin.controls.multichoicerow_body_nohover_color     = {240, 240, 240, 255}
skin.controls.multichoicerow_body_hover_color       = {225, 190, 25, 255}
skin.controls.multichoicerow_text_nohover_color     = {150, 150, 150, 255}
skin.controls.multichoicerow_text_hover_color       = {255, 255, 255, 255}
skin.controls.multichoicerow_text_font              = smallfont

-- tooltip
skin.controls.tooltip_body_color                    = {255, 255, 255, 255}

-- textinput
skin.controls.textinput_body_color                  = {240, 240, 240, 255}
skin.controls.textinput_indicator_color             = {0, 0, 0, 50}
skin.controls.textinput_text_empty_color            = {210, 210, 210, 255}
skin.controls.textinput_text_normal_color           = {100, 100, 100, 255}
skin.controls.textinput_text_selected_color         = {255, 255, 255, 255}
skin.controls.textinput_highlight_bar_color         = {51, 204, 255, 255}

-- slider
skin.controls.slider_bar_outline_color              = {220, 220, 220, 10}

-- checkbox
skin.controls.checkbox_body_color                   = {255, 255, 255, 255}
skin.controls.checkbox_check_color                  = {88, 182, 35, 255}
skin.controls.checkbox_text_font                    = smallfont

-- collapsiblecategory
skin.controls.collapsiblecategory_text_color        = {0, 0, 0, 255}

-- columnlist
skin.controls.columnlist_body_color                 = {232, 232, 232, 100}

-- columlistarea
skin.controls.columnlistarea_body_color             = {232, 232, 232, 100}

-- columnlistheader
skin.controls.columnlistheader_text_down_color      = {100, 100, 100, 255}
skin.controls.columnlistheader_text_nohover_color   = {160, 160, 160, 255}
skin.controls.columnlistheader_text_hover_color     = {255, 255, 255, 255}
skin.controls.columnlistheader_text_font            = GetFont("AlteHaas.ttf",20)

-- columnlistrow
skin.controls.columnlistrow_body1_color             = {245, 245, 245, 255}
skin.controls.columnlistrow_body2_color             = {255, 255, 255, 255}
skin.controls.columnlistrow_body_selected_color     = {255, 255, 255, 160}
skin.controls.columnlistrow_body_hover_color        = {255, 255, 255, 50}
skin.controls.columnlistrow_text_color              = {100, 100, 100, 255}
skin.controls.columnlistrow_text_hover_color        = {255, 255, 255, 255}
skin.controls.columnlistrow_text_selected_color     = {255, 255, 255, 255}
skin.controls.columnlistrow_text_font               = GetFont("Sansation.ttf",20)

-- modalbackground
skin.controls.modalbackground_body_color            = {0, 0, 0, 140}

-- linenumberspanel
skin.controls.linenumberspanel_text_color           = {100, 100, 100, 255}
skin.controls.linenumberspanel_body_color			= {200, 200, 200, 255}

local obj = {}

function obj.GetBorder()
	return {143, 143, 143, 255}
end

--[[---------------------------------------------------------
	- func: OutlinedRectangle(object)
	- desc: creates and outlined rectangle
--]]---------------------------------------------------------
function skin.OutlinedRectangle(x, y, width, height, ovt, ovb, ovl, ovr)

	local ovt = ovt or false
	local ovb = ovb or false
	local ovl = ovl or false
	local ovr = ovr or false

	-- top
	if not ovt then
		love.graphics.rectangle("fill", x, y, width, 1)
	end

	-- bottom
	if not ovb then
		love.graphics.rectangle("fill", x, y + height - 1, width, 1)
	end

	-- left
	if not ovl then
		love.graphics.rectangle("fill", x, y, 1, height)
	end

	-- right
	if not ovr then
		love.graphics.rectangle("fill", x + width - 1, y, 1, height)
	end

end

function skin.DrawDisplayText(object)---Omar's function for drawing my text on frames and buttons
	local text = object.displayText;
	local textColor = object.displayTextColor or {0,0,0,255};
	local font = skin.controls.displayTextFont
	local f = 1--(_Screen.width/1920)

	if(text)then
		love.graphics.setFont(font)
		love.graphics.setColor(unpack(textColor))
		love.graphics.print(text,object.x + 6 * f,object.y + 3 * f,0,object.scaleX,object.scaleY)
		love.graphics.setColor(255,255,255,255)
	end
end

--[[---------------------------------------------------------
	- func: DrawFrame(object)
	- desc: draws the frame object
--]]---------------------------------------------------------
function skin.DrawFrame(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local hover = object:GetHover()
	local name = object:GetName()
	local icon = object:GetIcon()
	local bodycolor = skin.controls.frame_body_color
	local topcolor = skin.controls.frame_top_color
	local namecolor = skin.controls.frame_name_color
	local font = skin.controls.frame_name_font
	local image = skin.images["frame-topbar.png"]
	local imageheight = image:getHeight()
	local scaley = 25/imageheight



	if(object.mask)then
		local func = function()
			object.maskDebug(object.maskX,object.maskY,object.scaleX,object.scaleY)
		end
		local stencil = func
		love.graphics.setStencil(stencil)
	end

	local r = function()
		love.graphics.draw(object.sprite,object.quad or nil,object.x,object.y,object.angle,object.scaleX * object.OscaleX,object.scaleY * object.OscaleY,
			object.offsetx, object.offsety);
	end

	love.graphics.setColor(object.color or {255,255,255,255});

	r()

	if(object.whiten)then
		love.graphics.setBlendMode("additive")
		love.graphics.setColor(255,255,255,255 * object.whiteAlpha)
		r();
		love.graphics.setColor(255,255,255,255)
		love.graphics.setBlendMode("alpha")
	end

	skin.DrawDisplayText(object)


	if(object.mask)then
		love.graphics.setStencil()
	end

end

function skin.DrawCustomFrame(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local hover = object:GetHover()
	local name = object:GetName()
	local icon = object:GetIcon()
	local bodycolor = skin.controls.frame_body_color
	local topcolor = skin.controls.frame_top_color
	local namecolor = skin.controls.frame_name_color
	local font = skin.controls.frame_name_font
	local image = skin.images["frame-topbar.png"]
	local imageheight = image:getHeight()
	local scaley = 25/imageheight

	if(object.mask)then
		local func = function()
			object.maskDebug(object.maskX,object.maskY,object.scaleX,object.scaleY)
		end
		local stencil = func
		love.graphics.setStencil(stencil)
	end

	local factor = 1--_Screen.width/1920;
	local offY = 40 * factor;
	local objheight = object.height - offY
	local objy = object.y + offY;

	local x,y,w,h = object.quadArray["WindowCorner"]:getViewport()
	local delta = {
	{0,1,0},{1,1,-math.pi/2},{1,0,-math.pi},{0,0,math.pi/2}
	}

	local delta2 = {
		object.width - (w*2) * factor,objheight - (h*2) * factor,object.width- (w*2) * factor,objheight - (h*2) * factor
	}

	love.graphics.setColor(77,90,88,255);
	love.graphics.rectangle("fill",object.x + (w/2) * factor,objy + (h/2) * factor,object.width - w * factor,objheight - h * factor)
	love.graphics.setColor(255,255,255,255);


	for i=1,4 do
		local X,Y = delta[i][1] * object.width,delta[i][2] * objheight
		local fX = 1;
		local fY = 1;
		if(i == 1)then fY = -1 end
		if(i == 2)then fX = -1; fY = -1 end
		if(i == 3)then fX = -1 end

		X = X + object.x + (w/2) * object.scaleX * fX; Y = Y + objy + (h/2) * object.scaleY * fY
		love.graphics.draw(_Images["EditorUISheet"],object.quadArray["WindowCorner"],X,Y,delta[i][3],object.scaleX,object.scaleY,w/2,h/2)

		X = X + (w/2) * object.scaleX * fX; Y = Y + (h/2) * object.scaleY * fY
		love.graphics.draw(_Images["EditorUISheet"],object.quadArray["WindowLine"], X,Y,delta[i][3],delta2[i],object.scaleY);
	end

	local tabWidth = object.width - 70 * factor
	local X = object.x + object.width/2 - tabWidth * .5;
	local Y = object.y - 5 * factor;
	love.graphics.draw(_Images["EditorUISheet"],object.quadArray["WindowTabLine"],X,Y,0,tabWidth,object.scaleY)
	love.graphics.draw(_Images["EditorUISheet"],object.quadArray["WindowTabEdge"],X - 14 * factor,Y,0,object.scaleX,object.scaleY)
	love.graphics.draw(_Images["EditorUISheet"],object.quadArray["WindowTabEdge"],X + 14 * factor + tabWidth,Y,0,-object.scaleX,object.scaleY)

	if(object.mask)then
		love.graphics.setStencil()
	end

end

--[[---------------------------------------------------------
	- func: DrawButton(object)
	- desc: draws the button object
--]]---------------------------------------------------------
function skin.DrawButton(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local hover = object:GetHover()
	local text = object:GetText()
	local font = skin.controls.button_text_font
	local twidth = font:getWidth(object.text)
	local theight = font:getHeight(object.text)
	local down = object.down
	local enabled = object:GetEnabled()
	local clickable = object:GetClickable()
	local textdowncolor = skin.controls.button_text_down_color
	local texthovercolor = skin.controls.button_text_hover_color
	local textnohovercolor = skin.controls.button_text_nohover_color
	local textnonclickablecolor = skin.controls.button_text_nonclickable_color
	local text = object.text;
	local font = skin.controls.imagebutton_text_font

	local quad = object.quad;

	local extraAlpha = 1;
	if not enabled or not clickable then
		--Draw unclickable button
		extraAlpha = .2;
		down = false; hover = false;
	end

	if down then
		---Draw down frame
		quad = object.downQuad;

	elseif hover then
		---Draw hover
		quad = object.overQuad
	else
		--Draw normal
	end

	if(object.Active and down ~= true)then quad = object.activeQuad end

	if(object.mask)then
		local func = function()
			object.maskDebug(object.maskX,object.maskY,object.scaleX,object.scaleY)
		end
		local stencil = func
		love.graphics.setStencil(stencil)
	end

	if(object.isThumb)then
		love.graphics.draw(loveframes.util.thumbQuads["sheet"],loveframes.util.thumbQuads["normal"],object.x,object.y,object.angle,object.scaleX,object.scaleY);
	end

	local offx,offy = 0,0;
	if(object.myoffsetx ~= nil)then offx = object.myoffsetx end
	if(object.myoffsety ~= nil)then offy = object.myoffsety end

	local r = function()

		if(object.quad == nil)then
			love.graphics.draw(object.sprite,object.x + offx,object.y + offy,object.angle,object.scaleX * object.OscaleX,object.scaleY * object.OscaleY,object.offSetX,object.offSetY)
		else
			love.graphics.draw(object.sprite,quad,object.x + offx,object.y + offy,object.angle,object.scaleX * object.OscaleX,object.scaleY * object.OscaleY,object.offSetX,object.offSetY)
		end

	end



	love.graphics.setColor(255,255,255,255*object.alpha * extraAlpha)
	r();
	love.graphics.setColor(255,255,255,255)

	if(object.isThumb)then
		if(hover)then
			love.graphics.draw(loveframes.util.thumbQuads["sheet"],loveframes.util.thumbQuads["over"],object.x,object.y,object.angle,object.scaleX,object.scaleY);
		end
	end

	if(object.whiten)then
		love.graphics.setBlendMode("additive")
		love.graphics.setColor(255,255,255,255 * object.whiteAlpha)
		r();
		love.graphics.setColor(255,255,255,255)
		love.graphics.setBlendMode("alpha")
	end

	if(object.drawSelectedThing)then ---I know this is a really bad and manual thing but
		---dont judge me
		love.graphics.draw(object.sprite,object.selectedQuad,object.x,object.y,object.angle,object.scaleX,object.scaleY);
	end

	skin.DrawDisplayText(object)

	if(object.mask)then
		love.graphics.setStencil()
	end

end

--[[---------------------------------------------------------
	- func: DrawCloseButton(object)
	- desc: draws the close button object
--]]---------------------------------------------------------
function skin.DrawCloseButton(object)

	local x = object:GetX()
	local y = object:GetY()
	local parent = object.parent
	local parentwidth = parent:GetWidth()
	local hover = object:GetHover()
	local down = object.down
	local image = skin.images["close.png"]
	local bodydowncolor = skin.controls.closebutton_body_down_color
	local bodyhovercolor = skin.controls.closebutton_body_hover_color
	local bodynohovercolor = skin.controls.closebutton_body_nohover_color

	image:setFilter("nearest", "nearest")

	if down then
		-- button body
		love.graphics.setColor(bodydowncolor)
		love.graphics.draw(image, x, y)
	elseif hover then
		-- button body
		love.graphics.setColor(bodyhovercolor)
		love.graphics.draw(image, x, y)
	else
		-- button body
		love.graphics.setColor(bodynohovercolor)
		love.graphics.draw(image, x, y)
	end

	object:SetPos(parentwidth - 20, 4)
	object:SetSize(16, 16)

end

--[[---------------------------------------------------------
	- func: DrawImage(object)
	- desc: draws the image object
--]]---------------------------------------------------------
function skin.DrawImage(object)

	local x = object:GetX()
	local y = object:GetY()
	local orientation = object:GetOrientation()
	local scalex = object:GetScaleX()
	local scaley = object:GetScaleY()
	local offsetx = object:GetOffsetX()
	local offsety = object:GetOffsetY()
	local shearx = object:GetShearX()
	local sheary = object:GetShearY()
	local image = object.image
	local color = object.imagecolor

	if color then
		love.graphics.setColor(color)
		love.graphics.draw(image, x, y, orientation, scalex, scaley, offsetx, offsety, shearx, sheary)
	else
		love.graphics.setColor(255, 255, 255, 255)
		love.graphics.draw(image, x, y, orientation, scalex, scaley, offsetx, offsety, shearx, sheary)
	end

	if(object.whiten)then
		love.graphics.setBlendMode("additive")
		love.graphics.setColor(255,255,255,255 * object.whiteAlpha)
		love.graphics.draw(image, x, y, orientation, scalex, scaley, offsetx, offsety, shearx, sheary)
		love.graphics.setColor(255,255,255,255)
		love.graphics.setBlendMode("alpha")
	end

end

--[[---------------------------------------------------------
	- func: DrawImageButton(object)
	- desc: draws the image button object
--]]---------------------------------------------------------
function skin.DrawImageButton(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local text = object:GetText()
	local hover = object:GetHover()
	local image = object:GetImage()
	local down = object.down
	local font = skin.controls.imagebutton_text_font
	local twidth = font:getWidth(object.text)
	local theight = font:getHeight(object.text)
	local textdowncolor = skin.controls.imagebutton_text_down_color
	local texthovercolor = skin.controls.imagebutton_text_hover_color
	local textnohovercolor = skin.controls.imagebutton_text_nohover_color

	if down then
		if image then
			love.graphics.setColor(255, 255, 255, 255)
			love.graphics.draw(image, x + 1, y + 1)
		end
		love.graphics.setFont(font)
		love.graphics.setColor(0, 0, 0, 255)
		love.graphics.print(text, x + width/2 - twidth/2 + 1, y + height - theight - 5 + 1)
		love.graphics.setColor(textdowncolor)
		love.graphics.print(text, x + width/2 - twidth/2 + 1, y + height - theight - 6 + 1)
	elseif hover then
		if image then
			love.graphics.setColor(255, 255, 255, 255)
			love.graphics.draw(image, x, y)
		end
		love.graphics.setFont(font)
		love.graphics.setColor(0, 0, 0, 255)
		love.graphics.print(text, x + width/2 - twidth/2, y + height - theight - 5)
		love.graphics.setColor(texthovercolor)
		love.graphics.print(text, x + width/2 - twidth/2, y + height - theight - 6)
	else
		if image then
			love.graphics.setColor(255, 255, 255, 255)
			love.graphics.draw(image, x, y)
		end
		love.graphics.setFont(font)
		love.graphics.setColor(0, 0, 0, 255)
		love.graphics.print(text, x + width/2 - twidth/2, y + height - theight - 5)
		love.graphics.setColor(textnohovercolor)
		love.graphics.print(text, x + width/2 - twidth/2, y + height - theight - 6)
	end

end

--[[---------------------------------------------------------
	- func: DrawProgressBar(object)
	- desc: draws the progress bar object
--]]---------------------------------------------------------
function skin.DrawProgressBar(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local value = object:GetValue()
	local max = object:GetMax()
	local barwidth = object:GetBarWidth()
	local font = skin.controls.progressbar_text_font
	local text = value .. "/" ..max
	local twidth = font:getWidth(text)
	local theight = font:getHeight("a")
	local bodycolor = skin.controls.progressbar_body_color
	local barcolor = skin.controls.progressbar_bar_color
	local textcolor = skin.controls.progressbar_text_color
	local image = skin.images["progressbar.png"]
	local imageheight = image:getHeight()
	local scaley = height/imageheight

	-- progress bar body
	love.graphics.setColor(bodycolor)
	love.graphics.rectangle("fill", x, y, width, height)
	love.graphics.setColor(255, 255, 255, 255)
	love.graphics.draw(image, x, y, 0, barwidth, scaley)
	love.graphics.setFont(font)
	love.graphics.setColor(textcolor)
	love.graphics.print(text, x + width/2 - twidth/2, y + height/2 - theight/2)

	-- progress bar border
	love.graphics.setColor(bordercolor)
	skin.OutlinedRectangle(x, y, width, height)

end

--[[---------------------------------------------------------
	- func: DrawScrollArea(object)
	- desc: draws the scroll area object
--]]---------------------------------------------------------
function skin.DrawScrollArea(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local bartype = object:GetBarType()
	local bodycolor = skin.controls.scrollarea_body_color


end

--[[---------------------------------------------------------
	- func: DrawScrollBar(object)
	- desc: draws the scroll bar object
--]]---------------------------------------------------------

function skin.DrawScrollBar(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local dragging = object:IsDragging()
	local hover = object:GetHover()
	local bartype = object:GetBarType()
	local bodydowncolor = skin.controls.scrollbar_body_down_color
	local bodyhovercolor = skin.controls.scrollbar_body_hover_color
	local bodynohvercolor = skin.controls.scrollbar_body_nohover_color
	local alpha = (object.alpha or 1) * object.parent.alpha;
	local barwidth = object.parent.barWidth or skin.controls.scrollbar_width or 15

	love.graphics.setColor(25,25,25,255 * alpha)
	love.graphics.rectangle("fill",object.parent.x+(object.parent.width-barwidth)*0.5,object.parent.y,barwidth,object.parent.height)

	love.graphics.setColor(255,255,255,255 * alpha)


	love.graphics.rectangle("fill",x+(width-barwidth)*0.5,y,barwidth,height)
	love.graphics.setColor(255,255,255,255)

end

--[[---------------------------------------------------------
	- func: DrawScrollBody(object)
	- desc: draws the scroll body object
--]]---------------------------------------------------------
function skin.DrawScrollBody(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local bodycolor = skin.controls.scrollbody_body_color

	love.graphics.setColor(bodycolor)
	love.graphics.rectangle("fill", x, y, width, height)

end

--[[---------------------------------------------------------
	- func: DrawPanel(object)
	- desc: draws the panel object
--]]---------------------------------------------------------
function skin.DrawPanel(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local bodycolor = object.bodycolor or skin.controls.panel_body_color
	local cornerRadius = object.cornerRadius or skin.controls.list_corner_radius
	local backgroundimage = object.backgroundimage

	if bodycolor[4] > 0 then
		love.graphics.setColor(bodycolor)

		if backgroundimage then
			if cornerRadius then
				-- This probably won't work within scrollareas, but we shouldn't need it there
				love.graphics.setStencil(function()
					love.graphics.circle('fill', x + cornerRadius, y + cornerRadius, cornerRadius)
					love.graphics.circle('fill', x + cornerRadius, y + height - cornerRadius, cornerRadius)
					love.graphics.circle('fill', x + width - cornerRadius, y + cornerRadius, cornerRadius)
					love.graphics.circle('fill', x + width - cornerRadius, y + height - cornerRadius, cornerRadius)
					love.graphics.rectangle('fill', x + cornerRadius, y, width - cornerRadius * 2, cornerRadius)
					love.graphics.rectangle('fill', x + cornerRadius, y + height - cornerRadius, width - cornerRadius * 2, cornerRadius)
					love.graphics.rectangle('fill', x, y + cornerRadius, width, height - cornerRadius * 2)
				end)
			end

			love.graphics.draw(backgroundimage, x, y, 0, width / backgroundimage:getWidth(), height / backgroundimage:getHeight())

			if cornerRadius then
				love.graphics.setStencil()
			end
		elseif cornerRadius then
			loveframes.util.drawRoundedBox(x, y, width, height, cornerRadius)
		else
			love.graphics.rectangle("fill", x, y, width, height)
		end
	end

end

--[[---------------------------------------------------------
	- func: DrawList(object)
	- desc: draws the list object
--]]---------------------------------------------------------
function skin.DrawList(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local bodycolor = object.bodycolor or skin.controls.list_body_color
	local cornerRadius = object.cornerRadius or skin.controls.list_corner_radius

	if bodycolor[4] > 0 then
		love.graphics.setColor(bodycolor)
		if cornerRadius then
			loveframes.util.drawRoundedBox(x, y, width, height, cornerRadius)
		else
			love.graphics.rectangle("fill", x, y, width, height)
		end
	end
end

--[[---------------------------------------------------------
	- func: DrawList(object)
	- desc: used to draw over the object and its children
--]]---------------------------------------------------------
function skin.DrawOverList(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()

	--[[
	love.graphics.setColor(bordercolor)
	skin.OutlinedRectangle(x, y, width, height)
	--]]

end

--[[---------------------------------------------------------
	- func: DrawTabPanel(object)
	- desc: draws the tab panel object
--]]---------------------------------------------------------
function skin.DrawTabPanel(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local buttonheight = object:GetHeightOfButtons()
	local bodycolor = skin.controls.tabpanel_body_color

	love.graphics.setColor(bodycolor)
	love.graphics.rectangle("fill", x, y + buttonheight, width, height - buttonheight)
	love.graphics.setColor(bordercolor)
	skin.OutlinedRectangle(x, y + buttonheight - 1, width, height - buttonheight + 2)

end

--[[---------------------------------------------------------
	- func: DrawTabButton(object)
	- desc: draws the tab button object
--]]---------------------------------------------------------
function skin.DrawTabButton(object)

	local x = object:GetX()
	local y = object:GetY()
	local hover = object:GetHover()
	local text = object:GetText()
	local image = object:GetImage()
	local tabnumber = object:GetTabNumber()
	local parent = object:GetParent()
	local ptabnumber = parent:GetTabNumber()
	local font = skin.controls.tab_text_font
	local twidth = font:getWidth(object.text)
	local theight = font:getHeight(object.text)
	local imagewidth = 0
	local imageheight = 0
	local texthovercolor = skin.controls.button_text_hover_color
	local textnohovercolor = skin.controls.button_text_nohover_color

	if image then
		image:setFilter("nearest", "nearest")
		imagewidth = image:getWidth()
		imageheight = image:getHeight()
		object.width = imagewidth + 15 + twidth
		if imageheight > theight then
			parent:SetTabHeight(imageheight + 5)
			object.height = imageheight + 5
		else
			object.height = parent.tabheight
		end
	else
		object.width = 10 + twidth
		object.height = parent.tabheight
	end

	local width  = object:GetWidth()
	local height = object:GetHeight()

	if tabnumber == ptabnumber then
		-- button body
		local gradient = skin.images["button-hover.png"]
		local gradientheight = gradient:getHeight()
		local scaley = height/gradientheight
		love.graphics.setColor(255, 255, 255, 255)
		love.graphics.draw(gradient, x, y, 0, width, scaley)
		-- button border
		love.graphics.setColor(bordercolor)
		skin.OutlinedRectangle(x, y, width, height)
		if image then
			-- button image
			love.graphics.setColor(255, 255, 255, 255)
			love.graphics.draw(image, x + 5, y + height/2 - imageheight/2)
			-- button text
			love.graphics.setFont(font)
			love.graphics.setColor(texthovercolor)
			love.graphics.print(text, x + imagewidth + 10, y + height/2 - theight/2)
		else
			-- button text
			love.graphics.setFont(font)
			love.graphics.setColor(texthovercolor)
			love.graphics.print(text, x + 5, y + height/2 - theight/2)
		end
	else
		-- button body
		local gradient = skin.images["button-nohover.png"]
		local gradientheight = gradient:getHeight()
		local scaley = height/gradientheight
		love.graphics.setColor(255, 255, 255, 255)
		love.graphics.draw(gradient, x, y, 0, width, scaley)
		-- button border
		love.graphics.setColor(bordercolor)
		skin.OutlinedRectangle(x, y, width, height)
		if image then
			-- button image
			love.graphics.setColor(255, 255, 255, 150)
			love.graphics.draw(image, x + 5, y + height/2 - imageheight/2)
			-- button text
			love.graphics.setFont(font)
			love.graphics.setColor(textnohovercolor)
			love.graphics.print(text, x + imagewidth + 10, y + height/2 - theight/2)
		else
			-- button text
			love.graphics.setFont(font)
			love.graphics.setColor(textnohovercolor)
			love.graphics.print(text, x + 5, y + height/2 - theight/2)
		end
	end

end

--[[---------------------------------------------------------
	- func: DrawMultiChoice(object)
	- desc: draws the multi choice object
--]]---------------------------------------------------------
function skin.DrawMultiChoice(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local text = object:GetText()
	local choice = object:GetChoice()
	local image = skin.images["multichoice-arrow.png"]
	local font = skin.controls.multichoice_text_font
	local theight = font:getHeight("a")
	local bodycolor = skin.controls.multichoice_body_color
	local textcolor = skin.controls.multichoice_text_color
	local enabled = object:GetEnabled()

	image:setFilter("nearest", "nearest")

	love.graphics.setColor(bodycolor)
	love.graphics.rectangle("fill", x + 1, y + 1, width - 2, height - 2)

	love.graphics.setColor(textcolor)
	love.graphics.setFont(font)

	--hereomar
	local scale = 1--_Screen.width/1920;
	local offy = (1-scale) * (height/2)

	if choice == "" then
		love.graphics.print(text, x + 5, y + height/2 - theight/2 + offy,0,scale,scale)
	else
		local textToPrint = choice;
		for n=1, #textToPrint do
			local txt = textToPrint:sub(1, n);
			local w = width;
			if(font:getWidth(txt) > w - 30)then
				textToPrint = txt .. "...";
				break
			end
		end
		love.graphics.print(textToPrint, x + 5, y + height/2 - theight/2 + offy,0,scale,scale)
	end

	love.graphics.setColor(255, 255, 255, 255)
	love.graphics.draw(image, x + width - 20, y + 5)

	love.graphics.setColor(bordercolor)
	skin.OutlinedRectangle(x, y, width, height)

	if(not enabled)then
		love.graphics.setColor(0,0,0,155)
		love.graphics.rectangle("fill", x, y , width , height)
	end

end

--[[---------------------------------------------------------
	- func: DrawMultiChoiceList(object)
	- desc: draws the multi choice list object
--]]---------------------------------------------------------
function skin.DrawMultiChoiceList(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local bodycolor = skin.controls.multichoicelist_body_color

	love.graphics.setColor(bodycolor)
	love.graphics.rectangle("fill", x, y, width, height)

end

--[[---------------------------------------------------------
	- func: DrawOverMultiChoiceList(object)
	- desc: draws over the multi choice list object
--]]---------------------------------------------------------
function skin.DrawOverMultiChoiceList(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()

	love.graphics.setColor(bordercolor)
	skin.OutlinedRectangle(x, y - 1, width, height + 1)

end

--[[---------------------------------------------------------
	- func: DrawMultiChoiceRow(object)
	- desc: draws the multi choice row object
--]]---------------------------------------------------------
function skin.DrawMultiChoiceRow(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local text = object:GetText()
	local font = skin.controls.multichoicerow_text_font
	local bodyhovecolor = skin.controls.multichoicerow_body_hover_color
	local texthovercolor = skin.controls.multichoicerow_text_hover_color
	local bodynohovercolor = skin.controls.multichoicerow_body_nohover_color
	local textnohovercolor = skin.controls.multichoicerow_text_nohover_color

	love.graphics.setFont(font)

	if object.hover then
		love.graphics.setColor(bodyhovecolor)
		love.graphics.rectangle("fill", x, y, width, height)
		love.graphics.setColor(texthovercolor)
		local textToPrint = text;
		for n=1, #textToPrint do
			local txt = textToPrint:sub(1, n);
			local w = width;
			if(font:getWidth(txt) > w - 30)then
				textToPrint = txt .. "...";
				break
			end
		end
		love.graphics.print(textToPrint, x + 5, y + 5)
	else
		love.graphics.setColor(bodynohovercolor)
		love.graphics.rectangle("fill", x, y, width, height)
		love.graphics.setColor(textnohovercolor)
		local textToPrint = text;
		for n=1, #textToPrint do
			local txt = textToPrint:sub(1, n);
			local w = width;
			if(font:getWidth(txt) > w - 30)then
				textToPrint = txt .. "...";
				break
			end
		end
		love.graphics.print(textToPrint, x + 5, y + 5)
	end

end

--[[---------------------------------------------------------
	- func: DrawToolTip(object)
	- desc: draws the tool tip object
--]]---------------------------------------------------------
function skin.DrawToolTip(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local bodycolor = skin.controls.tooltip_body_color

	love.graphics.setColor(bodycolor)
	love.graphics.rectangle("fill", x, y, width, height)
	love.graphics.setColor(bordercolor)
	skin.OutlinedRectangle(x, y, width, height)

end

--[[---------------------------------------------------------
	- func: DrawText(object)
	- desc: draws the text object
--]]---------------------------------------------------------
function skin.DrawText(object)

end

--[[---------------------------------------------------------
	- func: DrawTextInput(object)
	- desc: draws the text input object
--]]---------------------------------------------------------
function skin.DrawTextInput(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local font = object:GetFont()
	local focus = object:GetFocus()
	local showindicator = object:GetIndicatorVisibility()
	local alltextselected = object:IsAllTextSelected()
	local textx = object:GetTextX()
	local texty = object:GetTextY()
	local text = object:GetText()
	local multiline = object:GetMultiLine()
	local lines = object:GetLines()
	local offsetx = object:GetOffsetX()
	local offsety = object:GetOffsetY()
	local indicatorx = object:GetIndicatorX()
	local indicatory = object:GetIndicatorY()
	local vbar = object:HasVerticalScrollBar()
	local hbar = object:HasHorizontalScrollBar()
	local linenumbers = object:GetLineNumbersEnabled()
	local itemwidth = object:GetItemWidth()
	local theight = font:getHeight("a")
	local bodycolor = object.bodycolor or skin.controls.textinput_body_color
	local textnormalcolor = object.textcolor or skin.controls.textinput_text_normal_color
	local textemptycolor = object.emptyColor or skin.controls.textinput_text_empty_color
	local textselectedcolor = skin.controls.textinput_text_selected_color
	local highlightbarcolor = skin.controls.textinput_highlight_bar_color
	local indicatorcolor = object.indicatorcolor or skin.controls.textinput_indicator_color
	local enabled = object.enabled;
	local cornerRadius = object.cornerRadius or skin.controls.textinput_corner_radius

	if(object.hidebox == false)then
		love.graphics.setColor(bodycolor)
		if cornerRadius then
			loveframes.util.drawRoundedBox(x, y, width, height, cornerRadius)
		else
			love.graphics.rectangle("fill", x, y, width, height)
		end
	end

	local stencilfunc = function()
		love.graphics.rectangle("fill", object.x + object.textoffsetx, object.y, object.width - object.textoffsetx, object.height)
	end

	if object.vbar and object.hbar then
		stencilfunc = function() love.graphics.rectangle("fill", object.x, object.y, object.width - 16, object.height - 16) end
	end

	love.graphics.setStencil(stencilfunc)

	local center = (object.align == "center");
	local scale = 1--_Screen.width/1920;

	if(center)then ----this only works if not multi line
		local textWidth = object:getTextWidth(lines[1])
		textx = textx + width * .5 * scale - textWidth * .5 * scale
		indicatorx = indicatorx + width * .5 * scale - textWidth * .5 * scale;
	end

	if alltextselected then
		local bary = 0
		if multiline then
			for i=1, #lines do
				local twidth = font:getWidth(lines[i])
				if twidth == 0 then
					twidth = 5
				end
				love.graphics.setColor(highlightbarcolor)
				love.graphics.rectangle("fill", textx, texty + bary, twidth, theight)
				bary = bary + theight
			end
		else
			local twidth = object:getTextWidth(text) * scale
			love.graphics.setColor(highlightbarcolor)
			love.graphics.rectangle("fill", textx, texty, twidth, theight)
		end
	end

	if showindicator and focus then
		love.graphics.setColor(indicatorcolor)
		love.graphics.rectangle("fill", indicatorx, indicatory + 2, 3, theight - 2)
	end

	if(focus)then
		--print(showindicator,text,indicatorx)
	end

	if not multiline then
		object:SetTextOffsetY(height/2 - theight/2)
		if offsetx ~= 0 then
			--object:SetTextOffsetX(0)
		else
			--object:SetTextOffsetX(5)
		end
	else
		if vbar then
			if offsety ~= 0 then
				if hbar then
					--object:SetTextOffsetY(5)
				else
					--object:SetTextOffsetY(-5)
				end
			else
				--object:SetTextOffsetY(5)
			end
		else
			--object:SetTextOffsetY(5)
		end

		if hbar then
			if offsety ~= 0 then
				if linenumbers then
					local panel = object:GetLineNumbersPanel()
					if vbar then
						--object:SetTextOffsetX(5)
					else
						--object:SetTextOffsetX(-5)
					end
				else
					if vbar then
						--object:SetTextOffsetX(5)
					else
						--object:SetTextOffsetX(-5)
					end
				end
			else
				--object:SetTextOffsetX(5)
			end
		else
			--object:SetTextOffsetX(5)
		end

	end


	love.graphics.setFont(font)

	local empty = (#lines == 1 and string.len(lines[1]) == 0)

	if empty and object.emptyText then
		local c = textemptycolor
		love.graphics.setColor(c[1], c[2], c[3], c[4] * object.alpha)
		love.graphics.print(object.emptyText, textx, texty, 0, scale, scale)
	elseif alltextselected then
		love.graphics.setColor(textselectedcolor)
	else
		love.graphics.setColor(textnormalcolor[1],textnormalcolor[2],textnormalcolor[3],textnormalcolor[4] * object.alpha)
	end

	if multiline then
		for i=1, #lines do
			love.graphics.print(lines[i], textx, texty + theight * i - theight)
		end
	else
		local s = lines[1]
		if object.passwordChar then s = string.rep(object.passwordChar, string.len(s)) end
		love.graphics.print(s, textx, texty,0,scale,scale)
	end


	----Draw the inactive input field------
	if(enabled == false)then
		love.graphics.setColor(115,125,125,255)
		local twidth = font:getWidth(text) * scale
		love.graphics.rectangle("fill", x, y, width, height)
	end

	love.graphics.setColor(255,255,255,255)

end

--[[---------------------------------------------------------
	- func: DrawOverTextInput(object)
	- desc: draws over the text input object
--]]---------------------------------------------------------
function skin.DrawOverTextInput(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()

	if(object.hidebox == false and not object.hideborder)then
		love.graphics.setColor(bordercolor)
		skin.OutlinedRectangle(x, y, width, height)
	end

end

--[[---------------------------------------------------------
	- func: DrawScrollButton(object)
	- desc: draws the scroll button object
--]]---------------------------------------------------------
function skin.DrawScrollButton(object)

	if not loveframes.util.EditorQuads then return end

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local hover = object:GetHover()
	local scrolltype = object:GetScrollType()
	local down = object.down
	local bodydowncolor = skin.controls.button_body_down_color
	local bodyhovercolor = skin.controls.button_body_hover_color
	local bodynohovercolor = skin.controls.button_body_nohover_color

	local quad = "ScrollArrow_N"

	if(down or hover)then quad = "ScrollArrow_O" end

	local angle = 0;
	local xoff = 0;
	local yoff = 0;

	if scrolltype == "up" then
		angle = 0;
	elseif scrolltype == "down" then
		angle = math.pi;
		xoff = width;
		yoff = height;
	elseif scrolltype == "left" then
		angle = -math.pi/2
	elseif scrolltype == "right" then
		angle = math.pi/2
	end

	local bar = object.parent.parent:GetScrollBar()

	love.graphics.setColor(255,255,255,255 * bar.parent.alpha)

	sprite = _Images["EditorUISheet"];
	love.graphics.draw(sprite,loveframes.util.EditorQuads[quad],x + xoff,y+yoff,angle)

	love.graphics.setColor(255,255,255,255)


end

--[[---------------------------------------------------------
	- func: skin.DrawSlider(object)
	- desc: draws the slider object
--]]---------------------------------------------------------
function skin.DrawSlider(object)

	local bordercolor = obj.GetBorder();

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local slidtype = object:GetSlideType()
	local baroutlinecolor = skin.controls.slider_bar_outline_color
	local enabled = object:GetEnabled()

	local alpha = 1;

	if(not enabled)then alpha = .2 end

	if slidtype == "horizontal" then

		love.graphics.setColor(25,25,25,255)
		local thickness = 15;
		love.graphics.rectangle("fill",x,y + height/2 - thickness/2, width, thickness)

		---------------------------
	elseif slidtype == "vertical" then
		love.graphics.setColor(baroutlinecolor[1],baroutlinecolor[2],baroutlinecolor[3],baroutlinecolor[4] * alpha)
		love.graphics.rectangle("fill", x + width/2 - 5, y, 10, height)
		love.graphics.setColor(bordercolor[1],bordercolor[2],bordercolor[3],bordercolor[4] * alpha)
		love.graphics.rectangle("fill", x + width/2, y + 5, 1, height - 10)
	end

end

--[[---------------------------------------------------------
	- func: skin.DrawSliderButton(object)
	- desc: draws the slider button object
--]]---------------------------------------------------------
function skin.DrawSliderButton(object)

	local bordercolor = obj.GetBorder();

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local hover = object:GetHover()
	local down = object.down
	local parent = object:GetParent()
	local enabled = parent:GetEnabled()
	local bodydowncolor = skin.controls.button_body_down_color
	local bodyhovercolor = skin.controls.button_body_hover_color
	local bodynohvercolor = skin.controls.button_body_nohover_color

	if(enabled)then
		love.graphics.setColor(255,255,255,255)
	else
		love.graphics.setColor(55,55,55,255)
	end
	love.graphics.rectangle("fill",parent.x,parent.y+13,(x-parent.x)+width/2,15)

end
--[[---------------------------------------------------------
	- func: skin.DrawCheckBox(object)
	- desc: draws the check box object
--]]---------------------------------------------------------
function skin.DrawCheckBox(object)
	if object.hide then return end

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetBoxWidth()
	local height = object:GetBoxHeight()
	local checked = object:GetChecked()
	local hover = object:GetHover()
	local bodycolor = skin.controls.checkbox_body_color
	local checkcolor = skin.controls.checkbox_check_color
	local enabled = object:GetEnabled()

--	if not enabled then
--		love.graphics.setColor(220,220,220,55)
--		love.graphics.rectangle("fill", x, y, width, height)
--		love.graphics.setColor(150,150,150,55)
--		skin.OutlinedRectangle(x, y, width, height)

--		love.graphics.setColor(0,0,0,20)
--		love.graphics.rectangle("fill", x + 4, y + 4, width - 8, height - 8)
--		return;
--	end

--	love.graphics.setColor(bodycolor)
--	love.graphics.rectangle("fill", x, y, width, height)

--	love.graphics.setColor(bordercolor)
--	skin.OutlinedRectangle(x, y, width, height)

	local q = imgmanip:StripImages("MainMenuSheet",0,0,"UI",{});
	local menuQuads = q.quadArray;

	if checked then
		love.graphics.draw(_Images["MainMenuSheet"],menuQuads["Checkbox_On"],x,y,0,width/29,height/29)
	else
		love.graphics.draw(_Images["MainMenuSheet"],menuQuads["Checkbox_Off"],x,y,0,width/29,height/29)
	end


--	if checked then
--		love.graphics.setColor(checkcolor)
--		love.graphics.rectangle("fill", x + 4, y + 4, width - 8, height - 8)
--	end

--	if hover then
--		love.graphics.setColor(bordercolor)
--		skin.OutlinedRectangle(x + 4, y + 4, width - 8, height - 8)
--	end

end

--[[---------------------------------------------------------
	- func: skin.DrawCollapsibleCategory(object)
	- desc: draws the collapsible category object
--]]---------------------------------------------------------
function skin.DrawCollapsibleCategory(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local text = object:GetText()
	local textcolor = skin.controls.collapsiblecategory_text_color
	local font = smallfont
	local image = skin.images["button-nohover.png"]
	local imageheight = image:getHeight()
	local scaley = height/imageheight

	love.graphics.setColor(255, 255, 255, 255)
	love.graphics.draw(image, x, y, 0, width, scaley)

	love.graphics.setFont(font)
	love.graphics.setColor(textcolor)
	love.graphics.print(text, x + 5, y + 5)

	love.graphics.setColor(bordercolor)
	skin.OutlinedRectangle(x, y, width, height)

end



--[[---------------------------------------------------------
	- func: skin.DrawColumnList(object)
	- desc: draws the column list object
--]]---------------------------------------------------------
function skin.DrawColumnList(object)

	local bordercolor = obj.GetBorder();

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local bodycolor = skin.controls.columnlist_body_color

	---get if it has a scrollbar
	local internals = object.internals
	local list = internals[1]
	local scrollbar = list:GetScrollBar()
	local offWidth = 0;
	if(scrollbar ~= false)then offWidth = scrollbar:GetWidth() end

	-- love.graphics.setColor(bodycolor)
	-- love.graphics.rectangle("fill", x, y, width-offWidth, height)

end

--[[---------------------------------------------------------
	- func: skin.DrawColumnListHeader(object)
	- desc: draws the column list header object
--]]---------------------------------------------------------
function skin.DrawColumnListHeader(object)

	local bordercolor = obj.GetBorder();

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local hover = object:GetHover()
	local name = object:GetName()
	local down = object.down
	local font = object.parent.headerFont or skin.controls.columnlistheader_text_font
	local twidth = font:getWidth(object.name)
	local theight = font:getHeight(object.name)
	local bodydowncolor = skin.controls.columnlistheader_body_down_color
	local textdowncolor = skin.controls.columnlistheader_text_down_color
	local bodyhovercolor = skin.controls.columnlistheader_body_hover_color
	local textdowncolor = skin.controls.columnlistheader_text_hover_color
	local nohovercolor = skin.controls.columnlistheader_body_nohover_color
	local textnohovercolor = {255,255,255,255}
	local center = object.center;
	local offset = object.offset;
	if(offset == nil)then offset = 0 end

	if down then
		love.graphics.setColor({255,255,255,155})
		love.graphics.rectangle("fill",x,y,width,height)
		-- header name
		love.graphics.setFont(font)
		love.graphics.setColor(textnohovercolor)
		local X = x + width/2 - twidth/2;
		if(center ~= true)then X = x + offset end
		love.graphics.print(name, X, y + height/2 - theight/2)
	elseif hover then
		love.graphics.setColor({255,255,255,55})
		love.graphics.rectangle("fill",x,y,width,height)
		-- header name
		love.graphics.setFont(font)
		love.graphics.setColor(textnohovercolor)
		local X = x + width/2 - twidth/2;
		if(center ~= true)then X = x + offset end
		love.graphics.print(name, X, y + height/2 - theight/2)

	else
		-- header name
		love.graphics.setFont(font)
		love.graphics.setColor(textnohovercolor)
		local X = x + width/2 - twidth/2;
		if(center ~= true)then X = x + offset end
		love.graphics.print(name, X, y + height/2 - theight/2)
		love.graphics.setColor(bordercolor)
	end

end

--[[---------------------------------------------------------
	- func: skin.DrawColumnListArea(object)
	- desc: draws the column list area object
--]]---------------------------------------------------------
function skin.DrawColumnListArea(object)

	local bordercolor = obj.GetBorder();

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local bodycolor = skin.controls.columnlistarea_body_color

	love.graphics.setColor(255,255,255,255)
	love.graphics.setLineWidth(1)
	local columnHeight = object.parent.columnheight;
	love.graphics.rectangle("line", x, y+columnHeight, width, height-columnHeight)

end

--[[---------------------------------------------------------
	- func: skin.DrawOverColumnListArea(object)
	- desc: draws over the column list area object
--]]---------------------------------------------------------
function skin.DrawOverColumnListArea(object)

	local bordercolor = obj.GetBorder();

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()

	love.graphics.setColor(bordercolor)

end

--[[---------------------------------------------------------
	- func: skin.DrawColumnListRow(object)
	- desc: draws the column list row object
--]]---------------------------------------------------------
function skin.DrawColumnListRow(object)

	local bordercolor = obj.GetBorder();

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()
	local colorindex = object:GetColorIndex()
	local font = skin.controls.columnlistrow_text_font
	local columndata = object:GetColumnData()
	local textx = object:GetTextX()
	local texty = object:GetTextY()
	local parent = object:GetParent()
	local cdimensionArray = parent:GetParent():GetAllColumnSize()
	local cAlignArray = parent:GetParent():GetAllColumnAlignment()
	local cOffsetArray = parent:GetParent():GetAllColumnOffset()
	local theight = font:getHeight("a")
	local hover = object:GetHover()
	local selected = object:GetSelected()
	local body1color = skin.controls.columnlistrow_body1_color;
	local body2color = skin.controls.columnlistrow_body2_color;
	local bodyhovercolor = skin.controls.columnlistrow_body_hover_color;
	local bodyselectedcolor = skin.controls.columnlistrow_body_selected_color;
	local textcolor = {255,255,255,255};
	local texthovercolor = skin.controls.columnlistrow_text_hover_color;
	local textselectedcolor = skin.controls.columnlistrow_text_selected_color;

	object:SetTextPos(5, height/2 - theight/2)

	if selected then
		love.graphics.setColor(bodyselectedcolor)
		love.graphics.rectangle("fill", x, y, width, height)
		love.graphics.setColor(bordercolor)
		if(parent.parent.disableInnerLines ~= true)then
			skin.OutlinedRectangle(x, y, width, height, true, false, true, true)
		end
	elseif hover then
		love.graphics.setColor(bodyhovercolor)
		love.graphics.rectangle("fill", x, y, width, height)
		love.graphics.setColor(bordercolor)
		if(parent.parent.disableInnerLines ~= true)then
			skin.OutlinedRectangle(x, y, width, height, true, false, true, true)
		end
	else
		-- love.graphics.setLineWidth(1)
		-- love.graphics.setColor(255,255,255,255)
		-- love.graphics.rectangle("line",x,y,width,height)
	end

	local scale = 1--_Screen.width/1920;
	local offy = (1-scale) * (theight/2)
	local columnNum = 1;

	for k, v in ipairs(columndata) do
		love.graphics.setFont(font)
		if selected then
			love.graphics.setColor(textselectedcolor)
		elseif hover then
			love.graphics.setColor(texthovercolor)
		else
			love.graphics.setColor(textcolor)
		end

		local w = cdimensionArray[columnNum][1];
		if w <2 then w = width end
		local textToPrint = truncate_by_width(v, w-5, font)

		local isSlotColumn = k == 8 and v and ( v == "0" or v == "1" or v == "2" or v == "3" or v == "4" );
		local isPassColumn = k == 3 and v == "+";
		local isPingColumn = k == 5 and tonumber(v);
		local isMutrColumn = k == 6 and v and ( v == TEXT("ABS") or v == TEXT("ABT") );

		local centerOff = 0;
		if(cAlignArray[columnNum])then
			local txtWidth = font:getWidth(textToPrint);
			centerOff = cdimensionArray[columnNum][1] - txtWidth;
		end

		love.graphics.setLineWidth(1)
		love.graphics.setColor(255,255,255,255)
		if(parent.parent.disableInnerLines ~= true)then
			love.graphics.rectangle("line",x,y,cdimensionArray[columnNum][1],height)
		end

		if isSlotColumn then
			local q = imgmanip:StripImages("MainMenuSheet",0,0,"UI",{});
			local menuQuads = q.quadArray;
			love.graphics.setColor(255,255,255,255)
			local quad = menuQuads["Slots_"..v]
			if quad then
				love.graphics.draw(_Images["MainMenuSheet"],quad,x+12,y+8,0)
			end
		elseif isPassColumn then
			local q = imgmanip:StripImages("MainMenuSheet",0,0,"UI",{});
			local menuQuads = q.quadArray;
			love.graphics.setColor(255,255,255,255)
			local quad = menuQuads["Password_On"]
			if quad then
				local qx, qy, qw, qh = quad:getViewport()
				love.graphics.draw(_Images["MainMenuSheet"], quad,
					x+(cdimensionArray[columnNum][1]-qw)/2,
					y+(cdimensionArray[columnNum][2]-qh)/2)
			end
		elseif isMutrColumn then
			local q = imgmanip:StripImages("MainMenuSheet",0,0,"UI",{});
			local menuQuads = q.quadArray;
			local quad = menuQuads["IconApply"]
			if v == TEXT("ABS") then
				love.graphics.setColor(195, 245, 40)
			else
				love.graphics.setColor(240, 85, 90)
				quad = menuQuads["IconDeny"]
			end
			if quad then
				local qx, qy, qw, qh = quad:getViewport()
				local scale = 0.8
				love.graphics.draw(_Images["MainMenuSheet"], quad,
					x+(cdimensionArray[columnNum][1]-(qw*scale))/2,
					y+(cdimensionArray[columnNum][2]-(qh*scale))/2, 0, scale)
			end
		elseif cdimensionArray[columnNum][1] > 0 then
			if isPingColumn then
				local ping = tonumber(v)

				if ping <= 100 then
					color = {195, 245, 40}
				elseif ping <= 135 then
					color = {235, 185, 15}
				else
					color = {240, 85, 90}
				end

				love.graphics.setColor(color)
			end

			love.graphics.print(textToPrint, x + centerOff/2+ cOffsetArray[columnNum], y + texty + offy,0,scale,scale)
		end

		x = x + cdimensionArray[columnNum][1];
		columnNum = columnNum + 1;
	end

	if(object.parent.parent.hasTrash)then
		local quad = loveframes.util.EditorQuads["Trash_N"];
		local overRowQuad = loveframes.util.EditorQuads["Trash_A"];
		local overQuad = loveframes.util.EditorQuads["Trash_O"];
		local downQuad = loveframes.util.EditorQuads["Trash_D"];

		if(hover or selected)then quad = overRowQuad end

		local mouseIn = false;
		local mx, my = _cam2:worldCoords(love.mouse.getPosition())
		local diffx = mx - x
		local diffy = my - y;
		if(diffx > -20 and diffx < -8)then
			if(diffy > 1 and diffy < 20)then mouseIn = true end
		end

		if(mouseIn)then
			quad = overQuad
			if(_mouseButtons[1])then quad = downQuad end

		end

		object.mouseInTrash = mouseIn;

		love.graphics.setColor(255,255,255,255)
		love.graphics.draw(_Images["EditorUISheet"],quad,x-18,y+4,0)
	end

end

--[[---------------------------------------------------------
	- func: skin.DrawModalBackground(object)
	- desc: draws the modal background object
--]]---------------------------------------------------------
function skin.DrawModalBackground(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = 1920;
	local height = 1080;
	local bodycolor = skin.controls.modalbackground_body_color

	love.graphics.setColor(bodycolor)
	love.graphics.rectangle("fill", x, y, width, height)

end

--[[---------------------------------------------------------
	- func: skin.DrawLineNumbersPanel(object)
	- desc: draws the line numbers panel object
--]]---------------------------------------------------------
function skin.DrawLineNumbersPanel(object)



end

--[[---------------------------------------------------------
	- func: skin.DrawNumberBox(object)
	- desc: draws the numberbox object
--]]---------------------------------------------------------
function skin.DrawNumberBox(object)

end

--[[---------------------------------------------------------
	- func: skin.DrawGrid(object)
	- desc: draws the grid object
--]]---------------------------------------------------------
function skin.DrawGrid(object)

	local x = object:GetX()
	local y = object:GetY()
	local width = object:GetWidth()
	local height = object:GetHeight()

	love.graphics.setColor(bordercolor)
	skin.OutlinedRectangle(x, y, width, height)

	local cx = x
	local cy = y
	local cw = object.cellwidth + (object.cellpadding * 2)
	local ch = object.cellheight + (object.cellpadding * 2)

	for i=1, object.rows do
		for n=1, object.columns do
			local ovt = false
			local ovl = false
			if i > 1 then
				ovt = true
			end
			if n > 1 then
				ovl = true
			end
			skin.OutlinedRectangle(cx, cy, cw, ch, ovt, false, ovl, false)
			cx = cx + cw
		end
		cx = x
		cy = cy + ch
	end

end

-- register the skin
loveframes.skins.Register(skin)
